REBOL [
	Title:		"rebXR - XML-RPC library for REBOL, marshaling component"
	File:		%xmlrpc-marshaler.r

	Author: 	"Andreas Bolka"
	Email:		andreas.bolka@gmx.net
	Rights: {
		Copyright (C) 2001, 2002 by Andreas Bolka
		Licensed under the Academic Free License version 1.1. 

		See: <URL:http://earl.strain.at/license/afl> or
		<URL:http://www.rosenlaw.com/afl.html>
    }

	Tabs:		4
	
	CVS-Date:		"$Date: 2002/11/10 00:32:11 $"
	CVS-Revision:	"$Revision: 1.4 $"
]

xmlrpc-marshaler: make object! [

	timer: function [ block ] [
		t0 t1 res 
	] [
		t0: now/time/precise
		res: do block
		t1: now/time/precise
		print t1 - t0
		res
	]

	; this will be the xml output string
	; it is initialised by marshal-[resp|req]
	; if none? marshal-[val/array/struct/date] will return a string
	xml-str: none

	system/error: make system/error [
		xmlrpc: make object! [
			code: 1000
			type: "XML-RPC Fault"
			fault: [ "Code:" :arg1 "Message:" :arg2 ]
		]
	]

	unmarshal-resp: function [ xml-response [ string! ] ] [
		tree methodResponse fault
	] [
		tree: make object! ( xml-to-object parse-xml+ xml-response )
		methodResponse: tree/document/methodResponse

		resp-type: to-string next first methodResponse
		switch resp-type [
			"params"	[ unmarshal-val methodResponse/params/param/value ]
			"fault" 	[ unmarshal-fault methodResponse/fault/value ] 
		]
	]

	unmarshal-req: function [ xml-request [string!] ] [
		tree methodCall call param
	] [
		tree: make object! ( xml-to-object parse-xml+ xml-request )
		methodCall: tree/document/methodCall

		call: copy []
		; method name
		append call to-word methodCall/methodName/value?
		; params
		if = 'param (second first methodCall/params) [ ; non-empty params?
			foreach param compose [(methodCall/params/param)] [
				append/only call unmarshal-val param/value
			]
		]

		call
	]
	
	unmarshal-fault: function [ fault-node ] [
		fault e
	] [
		fault: unmarshal-struct fault-node/struct
		e: reduce [ 
			'xmlrpc 'fault select fault "faultCode" select fault "faultString" 
		]
		make error! e
	]

	unmarshal-val: function [ val-obj ] [
		type
	] [ 
		type: to-string next first val-obj
		switch type [
			; default: string
			"value?"	[ to-string any [ val-obj/value? "" ] ]
			; well-known basic xmlrpc types
			"string"	[ to-string any [ val-obj/string/value? "" ] ]
			"i4"		[ to-integer val-obj/i4/value? ]
			"int"		[ to-integer val-obj/int/value? ]
			"boolean"	[ to-logic to-integer val-obj/boolean/value? ]
			"double"	[ to-decimal val-obj/double/value? ]
			; advanced xmlrpc types
			"dateTime.iso8601" [ unmarshal-date val-obj/dateTime.iso8601 ]
			"base64"	[ to-binary debase/base val-obj/base64/value? 64 ]
			; complex xmlrpc types
			"array" 	[ unmarshal-array val-obj/array ]
			"struct" 	[ unmarshal-struct val-obj/struct ] 
		]
	]

	unmarshal-date: function [ date-obj ] [
		digits yy mm dd tm
	] [
		digits: charset [ #"0" - #"9" ]
		parse date-obj/value? [
			copy yy 4 digits copy mm 2 digits copy dd 2 digits
			"T" copy tm to end ]
		return make date! reduce [ load yy load mm load dd load tm ]
	]

	unmarshal-array: function [ val-obj ] [
		values array val
	] [
		if error? try [
			values: compose [(val-obj/data/value)]
			array: make block! length? values 
			foreach val values [
				append/only array unmarshal-val val 
			]
		] [
			array: copy []
		]

		array
	]

	unmarshal-struct: function [ val-obj ] [
		members struct member key val
	] [
		if error? try [ 
			members: compose [(val-obj/member)]
			struct: make hash! length? members
			foreach member members [
				key: member/name/value?
				val: unmarshal-val member/value
				repend struct [ key val ]
			]
		] [
			struct: make hash! []
		]

		reduce struct
	]

	; ----

	marshal-resp: function [ retparam ] [
		retval
	] [
		xml-str: make xml-text-writer [] ; formatting: enum-formatting-indented ]

		xml-str/write-start-document
		xml-str/write-start-element "methodResponse"	
		xml-str/write-start-element "params"
		xml-str/write-start-element "param"
		marshal-val retparam
		xml-str/write-full-end-element

		retval: xml-str/content
		xml-str: none
		retval
	]

	marshal-req: function [ method-block ] [
		param retval
	] [
		xml-str: make xml-text-writer [] ; formatting: enum-formatting-indented ]

		xml-str/write-start-document
		xml-str/write-start-element "methodCall"
		xml-str/write-element-string "methodName" to-string first method-block
		xml-str/write-start-element "params"

		foreach param reduce next method-block [
			xml-str/write-start-element "param" 
			marshal-val param
			xml-str/write-end-element
		]

		xml-str/write-full-end-element

		retval: xml-str/content
		xml-str: none
		retval
	]

	marshal-fault: function [ faultCode faultString ] [
		retval
	] [
		xml-str: make xml-text-writer [] ; formatting: enum-formatting-indented ]

		xml-str/write-start-document
		xml-str/write-start-element "methodResponse"
		xml-str/write-start-element "fault"
		marshal-val make hash! reduce [ "faultCode" faultCode "faultString" faultString ]
		xml-str/write-full-end-element

		retval: xml-str/content
		xml-str: none
		retval
	]
	
	marshal-val: function [ val ] [
		xml
	] [
		xml: xml-str

		xml/write-start-element "value" 
		switch to-string type? val [
			; well known basic xmlrpc types
			"string"	[ xml/write-element-string 	"string" 	val ]
			"integer"	[ xml/write-element-raw		"i4"	 	val ]
			"logic" 	[ xml/write-element-raw		"boolean"	(either val [1][0]) ]
			"decimal"	[ xml/write-element-raw 	"double" 	val ]
			; advanced xmlrpc types
			"date"		[ xml/write-element-raw 	"dateTime.iso8601" (marshal-date val) ]
			"binary"	[ xml/write-element-raw 	"base64" 	(enbase/base val 64) ]
			; complex xmlrpc types
			"block"		[ marshal-array	reduce val ]
			"hash" 		[ marshal-struct reduce val ]
		]
		xml/write-end-element

		xml/content
	]

	marshal-date: function [ date ] [
	] [
		pad-zero: func [ num ] [ either = 1 length? to-string num [ join "0" num ] [ num ] ]
		rejoin [ 
			date/year pad-zero date/month pad-zero date/day 
			"T" 
			pad-zero date/time/1 ":" pad-zero date/time/2 ":" pad-zero date/time/3 
		]
	]

	marshal-array: function [ array ] [
		xml elem
	] [
		xml: xml-str

		xml/write-start-element "array"
		xml/write-start-element "data"
		foreach elem array [
			marshal-val elem
		]
		xml/write-end-element
		xml/write-end-element

		xml/content
	]

	marshal-struct: function [ struct ] [
		xml key val
	] [
		xml: xml-str

		xml/write-start-element "struct"
		foreach [ key val ] struct [
			xml/write-start-element "member"
			xml/write-element-string "name" key
			marshal-val val
			xml/write-end-element
		]
		xml/write-end-element

		xml/content
	]

] ; xmlrpc-marshaler obj

; vim: set ts=4:
